import path from 'path';
import fsp from 'fs/promises';
import ky from 'ky';
import parseUrl from 'extract-tld';
import { spin } from "tiny-spin";
import { resolveCountry } from './resolveCountry.js';
import MirrorsByRegion from './mirrorsByRegion.js';
const inputFile = process.env['INPUT'] || path.join(process.cwd(), 'mirrors.json');
const fixtureFile = process.env['FIXTURE'] || path.join(process.cwd(), 'mirrors.fixture.json');
const mirrorList = process.env['MIRRORLIST'] || path.join(process.cwd(), 'mirrorlist');
const verbose = !!process.env['VERBOSE'];
const skipIsoCheck = !!process.env['SKIPISO'];
const protocolId = {
    http: 1,
    rsync: 3,
    https: 5
};
let mirrorCounter = 0;
let mirrorUrlCounter = 0;
async function IsosPresent(url) {
    if (skipIsoCheck) {
        return false;
    }
    const pathTests = ['iso', 'isos', 'ISO', 'ISOs', 'ISOS'];
    for (let i = 0; i < pathTests.length; i++) {
        try {
            await ky.get(`${url}${pathTests[i]}/`, { timeout: 5000 });
            return true;
        }
        catch (err) {
            if (!!err && typeof err === 'object' && !('response' in err)) {
                console.error(`Failed to connect to ${url}`);
                if (verbose) {
                    console.error(err);
                }
                break;
            }
        }
    }
    return false;
}
function setMirrors(fixture) {
    const mirrors = {};
    fixture.filter(x => x.model === 'mirrors.Mirror').forEach((mirror) => {
        mirrorCounter = Math.max(mirrorCounter, mirror.pk);
        mirror.fields.active = false;
        mirror.fields.public = false;
        mirror.fields.isos = false;
        mirrors[mirror.fields.name] = mirror;
    });
    return mirrors;
}
function processUrl(url) {
    return {
        name: parseUrl.parseUrl(url)?.domain,
        full: url,
        protocol: url.split(':')[0],
        partial: url.split('$repo')[0]
    };
}
function getMirror(url) {
    return mirrors[url.name] = {
        pk: mirrors[url.name]?.pk || ++mirrorCounter,
        model: 'mirrors.Mirror',
        fields: {
            name: url.name,
            tier: -1,
            upstream: null,
            admin_email: '',
            alternate_email: '',
            public: true,
            active: false,
            isos: false,
            rsync_user: '',
            rsync_password: '',
            bug: null,
            notes: ''
        }
    };
}
async function updateMirror(m, profile, url) {
    m.fields.tier = profile.tier;
    m.fields.admin_email = profile.admin_email || '';
    m.fields.alternate_email = profile.alternate_email || '';
    m.fields.notes = profile.notes || '';
    m.fields.active ||= profile.active;
    m.fields.public &&= profile.public;
    m.fields.isos ||= (skipIsoCheck && profile.isos) || (url.protocol !== 'rsync' && await IsosPresent(url.partial));
    if (url.protocol === 'rsync') {
        m.fields.rsync_user = profile.rsync_user || '';
        m.fields.rsync_password = profile.rsync_password || '';
    }
}
async function processMirrorProfile(m) {
    const url = processUrl(m.url);
    const mirror = getMirror(url);
    await updateMirror(mirror, m, url);
    const mirrorUrl = {
        pk: mirrorUrlCounter++,
        model: 'mirrors.MirrorUrl',
        fields: {
            url: url.partial,
            protocol: protocolId[url.protocol],
            mirror: mirror.pk,
            country: resolveCountry(m.country)?.iso2 || null,
            // populate ip fields with `mirrorresolv`
            has_ipv4: false,
            has_ipv6: false,
            active: m.active
        }
    };
    mirrorUrls.push(mirrorUrl);
}
function composeMirrorFixture() {
    const fixture = [];
    for (let mirrorName in mirrors) {
        fixture.push(mirrors[mirrorName]);
    }
    fixture.push.apply(fixture, mirrorUrls);
    return fixture;
}
function getDateTime() {
    const now = new Date();
    return now.toISOString().split('T')[0];
}
function generateMirrorlist(mirrors = []) {
    const httpMirrors = mirrors.filter(m => processUrl(m.url).protocol !== 'rsync');
    const mirrorsByRegion = new MirrorsByRegion(mirrors);
    const lines = [
        '##',
        '## Artix Linux repository mirrorlist',
        `## Generated on ${getDateTime()}`,
        '##',
        '',
        '# Artix mirrors',
        '# Use rankmirrors(1) to get a list of the fastest mirrors for your location,',
        '# e.g.: rankmirrors -v -n 5 /etc/pacman.d/mirrorlist',
        '# Then put the resulting list on top of this file.',
        '',
        '# Default mirrors'
    ];
    httpMirrors.filter(m => m.default).forEach(m => lines.push(`Server = ${m.url}`));
    lines.push('');
    mirrorsByRegion.printMirrors(lines);
    return lines.join('\n');
}
async function tryReadFileOrDefault(f, d) {
    try {
        return JSON.parse(await fsp.readFile(f, { encoding: 'utf-8' }));
    }
    catch (err) {
        console.error(`unable to open ${f}`);
        if (verbose) {
            console.error(err);
        }
        return d;
    }
}
const fixture = await tryReadFileOrDefault(fixtureFile, []);
const mirrors = setMirrors(fixture);
const mirrorUrls = [];
async function main() {
    const input = JSON.parse(await fsp.readFile(inputFile, { encoding: 'utf-8' }));
    const stopSpin = spin("Checking for ISOs...");
    for (let i = 0; i < input.mirrors?.length; i++) {
        await processMirrorProfile(input.mirrors[i]);
    }
    stopSpin();
    await fsp.writeFile(fixtureFile, JSON.stringify(composeMirrorFixture(), null, 4));
    await fsp.writeFile(mirrorList, generateMirrorlist(input.mirrors?.filter(m => m.public && m.active) || []));
}
export default main;
export { main };
//# sourceMappingURL=index.js.map